<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Broadcast;
use App\Models\BroadcastContact;
use App\Models\Message;
use Carbon\Carbon;

class BroadcastController extends Controller
{

    public function index()
    {
        $companyId = auth('api')->user()->company_id;

        return Broadcast::select(
            'broadcasts.*',
            'templates.template_name',
            \DB::raw('COUNT(broadcast_contacts.bc_id) AS contact_count')
        )
            ->leftJoin('templates', 'templates.template_id', '=', 'broadcasts.template_id')
            ->leftJoin('broadcast_contacts', 'broadcast_contacts.br_id', '=', 'broadcasts.br_id')
            ->where('broadcasts.company_id', $companyId)
            ->groupBy('broadcasts.br_id', 'templates.template_name')
            ->orderBy('broadcasts.br_id', 'DESC')
            ->get();
    }

    public function show($id)
    {
        return Broadcast::with('contacts')->findOrFail($id);
    }

    public function store(Request $request)
    {
        $request->validate([
            'broadcast_name' => 'required|string',
            'template_id' => 'nullable|integer',
            'type' => 'required|in:schedule,non-schedule',
            'schedule_date' => 'nullable|date',
            'schedule_time' => 'nullable',
            'contacts' => 'required|array'
        ]);
        $companyId = auth('api')->user()->company_id;
        $user_id = auth('api')->user()->id;

        // Make broadcast
        $broadcast = Broadcast::create([
            'broadcast_name'   => $request->broadcast_name,
            'template_id'      => $request->template_id,
            'type'             => $request->type,
            'date'           => Carbon::now()->format("Y-m-d"),
            'schedule_date'    => $request->schedule_date,
            'schedule_time'    => $request->schedule_time,
            'user_id'          => $user_id,
            'company_id'       => $companyId,
            'status'           => 'pending',
            'created_date_time' => Carbon::now(),
            'update_date_time' => Carbon::now(),
        ]);

        // Insert contacts
        foreach ($request->contacts as $contact) {
            BroadcastContact::create([
                'br_id'      => $broadcast->br_id,
                'contact_id' => $contact,
                'user_id'    => $user_id,
                'company_id' => $companyId,
                'status'     => 'pending'
            ]);
        }

        return response()->json([
            'message' => 'Broadcast created successfully.',
            'broadcast' => $broadcast
        ], 201);
    }


    public function update(Request $request, $id)
    {
        $broadcast = Broadcast::findOrFail($id);

        $broadcast->update(array_merge(
            $request->all(),
            ['update_date_time' => Carbon::now()]
        ));

        return response()->json([
            'message' => 'Broadcast updated successfully.',
            'broadcast' => $broadcast
        ]);
    }


    public function cancel($id)
    {
        $broadcast = Broadcast::findOrFail($id);

        $broadcast->status = 'cancelled';
        $broadcast->update_date_time = Carbon::now();
        $broadcast->save();

        return response()->json(['message' => 'Broadcast cancelled successfully.']);
    }


    public function addContact(Request $request)
    {
        $request->validate([
            'br_id' => 'required|integer',
            'contact_no' => 'required',
            'user_id' => 'required|integer',
            'company_id' => 'required|integer'
        ]);

        $contact = BroadcastContact::create([
            'br_id'      => $request->br_id,
            'contact_no' => $request->contact_no,
            'user_id'    => $request->user_id,
            'company_id' => $request->company_id,
            'status'     => 'pending'
        ]);

        return response()->json([
            'message' => 'Contact added.',
            'contact' => $contact
        ], 201);
    }

    public function deleteContact($contact_id)
    {
        BroadcastContact::destroy($contact_id);

        return response()->json([
            'message' => 'Broadcast contact deleted.'
        ]);
    }

    public function showdate(Request $request)
    {
        $fdate = $request->from_date;
        $tdate = $request->to_date;
        // $companyId = auth('api')->user()->company_id;

        $data = Broadcast::select(
            'broadcasts.*',
            'templates.template_name',
            \DB::raw('COUNT(broadcast_contacts.bc_id) AS contact_count')
        )
            ->leftJoin('templates', 'templates.template_id', '=', 'broadcasts.template_id')
            ->leftJoin('broadcast_contacts', 'broadcast_contacts.br_id', '=', 'broadcasts.br_id')
            ->whereBetween('broadcasts.date', [$fdate, $tdate])
            // ->where('broadcasts.company_id', $companyId)
            ->groupBy('broadcasts.br_id', 'templates.template_name')
            ->orderBy('broadcasts.br_id', 'DESC')
            ->get();

        if ($data->isEmpty()) {
            return response()->json([
                'status' => false,
                'message' => 'No broadcasts found for selected dates',
                'data'    => []
            ]);
        }

        return response()->json([
            'status' => true,
            'message' => 'Broadcasts loaded',
            'data'    => $data
        ]);
    }

    public function contacts($id)
    {
        $companyId = auth('api')->user()->company_id;

        $contacts = Message::where('messages.broadcast_id', $id)
            ->where('messages.company_id', $companyId)
            ->join('contacts', 'contacts.contact_id', '=', 'messages.contact_id')
            ->select(
                'contacts.mobile_no as contact_no',
                'contacts.contact_name as contact_name',
                'messages.message_status',
                'messages.created_date_time'
            )
            ->orderBy('messages.created_date_time', 'ASC')
            ->get();


        return response()->json([
            'success' => true,
            'contacts' => $contacts
        ]);
    }
}
