<?php

namespace App\Http\Controllers;

use App\Models\Message;
// use App\Models\User;
use App\Models\Company;
use App\Models\Contacts;
use App\Models\WhatsAppAccount;
// use Faker\Provider\bg_BG\PhoneNumber;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use PhpParser\Node\Stmt\Else_;
use Carbon\Carbon;
use Illuminate\Support\Collection;

class MessageController extends Controller
{

    public function receiveMessage(Request $request, $id)
    {
        $companyId = auth('api')->user()->company_id; // or $request->company_id;

        $messages = Message::where('contact_id', $id)
            ->where('company_id', $companyId)
            ->orderBy('created_date_time', 'desc')
            ->get();

        return response()->json($messages);
    }


    public function messages(Request $request, $id)
    {
        $companyId = auth('api')->user()->company_id;
        $messages = Message::where('contact_id', $id)
        ->where('company_id', $companyId)
        ->whereNot('type', 'broadcast') 
        ->where('created_date_time', '>=', Carbon::now()->subDay())
        ->orderBy('message_id', 'asc')
            ->get(['message_id', 'contact_id', 'message_type', 'wa_id', 'message', 'direction', 'contact_no','contact_name','mime_type', 'created_date_time', 'message_status']);

        $grouped = $messages->groupBy(function ($message) {
            $date = Carbon::parse($message->created_date_time);
            if ($date->isToday()) {
                return 'Today';
            } elseif ($date->isYesterday()) {
                return 'Yesterday';
            } else {
                return $date->format('M d, Y');
            }
        });

        $formatted = [];
        $idCounter = 1;

        foreach ($grouped as $day => $messagesForDay) {
            $formatted[] = [
                'id' => $idCounter++,
                'day' => $day,
                'messages' => $messagesForDay->values(),
                'created_date_time' => $messagesForDay->first()->created_date_time,
            ];
        }
        Message::where('contact_id', $id)
            ->where('company_id', $companyId)
            ->where('read_status', 'Unread')
            ->update(['read_status' => 'Read']);
        return response()->json($formatted);
    }

    public function sendMessage(Request $request)
    {
        $companyId = auth('api')->user()->company_id;

        $validated = $request->validate([
            'message' => 'required',
            'type' => 'in:chat,broadcast',
            'broadcast_id' => 'nullable',
            'phone_number' => 'nullable',
            'contact_id' => 'nullable'
        ]);
        if ($validated['contact_id'] == null) {
            $contact = Contacts::where('mobile_no', $validated['phone_number'])->first(['contact_id']);
            if ($contact) {
                $contact_id = $contact->contact_id;
            } else {
                $newContact = Contacts::create([
                    'mobile_no' => "91" . $validated['phone_number'],
                    'company_id' => $companyId ?? null,
                    'contact_name' => 'User'
                ]);
                $contact_id = $newContact->contact_id;
            }
        } else {
            $contact_id = $validated['contact_id'];
        }
        //  dd($contact_id);
        $message = Message::create([
            'company_id' => $companyId,
            'contact_id' => $contact_id,
            'type' => $validated['type'],
            'broadcast_id' => $validated['broadcast_id'] ?? null,
            'message' => $validated['message'],
            'direction' => 'out',
            'user_id' => auth()->id(),
            'message_status' => 'Sent',
            'read_status' => 'Sent',
            'wa_id' => '',
            'media_id' => '0',
            'created_date_time' => NOW(),
            'updated_date_time' => NOW(),
            'status' => 'Active'
        ]);
        $contact_number = Contacts::where('contact_id', $contact_id)->first(['mobile_no']);

        // $company = Company::where('user_id', auth()->id())->first(['company_id']);

        // if (!$company) {
        //     return response()->json(['error' => 'Company not found'], 404);
        // }

        $whatsappAccount = WhatsAppAccount::where('company_id', $companyId )->first();

        if (!$whatsappAccount) {
            return response()->json(['error' => 'WhatsApp account not found'], 404);
        }


        try {
            $response = Http::withHeaders([
                'Authorization' => "Bearer {$whatsappAccount->access_token}",
                'Content-Type' => 'application/json',
            ])
                ->withOptions([
                    'verify' => false,
                ])
                ->post("https://graph.facebook.com/v21.0/{$whatsappAccount->phone_number_id}/messages", [
                    'messaging_product' => 'whatsapp',
                    'recipient_type' => 'individual',
                    'to' =>  $contact_number->mobile_no,
                    'type' => 'text',
                    'text' => [
                        'body' => $validated['message']
                    ]
                ]);

            if ($response->successful()) {
                $whatsappResponse = $response->json();
                $messageId = $whatsappResponse['messages'][0]['id'] ?? null;

                if ($messageId) {
                    $message->wa_id = $messageId;
                    $message->message_status = 'Sent';
                    $message->save();
                }

                return response()->json([
                    'message' => $message,
                    'whatsapp_response' => $response->json()
                ], 201);
            } else {
                $message->update(['message_status' => 'Failed']);

                return response()->json([
                    'error' => 'Failed to send message via WhatsApp',
                    'whatsapp_response' => $response->json()
                ], 400);
            }
        } catch (\Exception $e) {
            $message->update(['message_status' => 'Failed']);

            return response()->json([
                'error' => 'Exception occurred while sending message',
                'message' => $e->getMessage()
            ], 500);
        }
    }
    public function sendMediaMessage(Request $request)
{
    $companyId = auth('api')->user()->company_id;
    $user_id = auth('api')->user()->id;

    $validated = $request->validate([
        'media' => 'required|file|max:102400',
        'caption' => 'nullable|string',
        'phone_number' => 'nullable',
        'contact_id' => 'nullable|integer',
        'broadcast_id' => 'nullable'
    ]);

    // Detect media type
    $mime = $request->file('media')->getMimeType();
    if (str_contains($mime, 'image')) $mediaType = 'image';
    elseif (str_contains($mime, 'video')) $mediaType = 'video';
    elseif (str_contains($mime, 'audio')) $mediaType = 'audio';
    else $mediaType = 'document';

    $image = $request->file('media');
    $destinationPath = base_path("../file_storage/sent_file/");
    $tempName = uniqid('temp_') . "." . $image->getClientOriginalExtension();
    $tempPath = $destinationPath . $tempName;
    $image->move($destinationPath, $tempName);

    // Step 1: Find or create contact
    if (empty($validated['contact_id'])) {
        $contact = Contacts::where('mobile_no', $validated['phone_number'])->first(['contact_id']);
        if ($contact) {
            $contact_id = $contact->contact_id;
        } else {
            $newContact = Contacts::create([
                'mobile_no' => "91" . $validated['phone_number'],
                'company_id' => $companyId ?? null,
                'contact_name' => 'User'
            ]);
            $contact_id = $newContact->contact_id;
        }
    } else {
        $contact_id = $validated['contact_id'];
    }

    $contact_number = Contacts::where('contact_id', $contact_id)->first(['mobile_no']);
    $whatsappAccount = WhatsAppAccount::where('company_id', $companyId)->first();
    if (!$whatsappAccount) {
        return response()->json(['error' => 'WhatsApp account not found'], 404);
    }

    try {
        // Step 2: Upload media to WhatsApp
        $uploadResponse = Http::withHeaders([
            'Authorization' => "Bearer {$whatsappAccount->access_token}",
        ])->withOptions(['verify' => false])
            ->attach('file', fopen($tempPath, 'r'), $image->getClientOriginalName())
            ->post("https://graph.facebook.com/v21.0/{$whatsappAccount->phone_number_id}/media", [
                'messaging_product' => 'whatsapp'
            ]);

        if (!$uploadResponse->successful()) {
            return response()->json([
                'error' => 'Failed to upload media to WhatsApp',
                'response' => $uploadResponse->json()
            ], 400);
        }

        $mediaId = $uploadResponse->json()['id'] ?? null;
        if (!$mediaId) {
            return response()->json(['error' => 'No media ID returned'], 400);
        }

        // Step 3: Create message in DB (Pending)
        $message = Message::create([
            'company_id' => $companyId,
            'contact_id' => $contact_id,
            'type' => 'chat',
            'broadcast_id' => $validated['broadcast_id'] ?? null,
            'message' => $validated['caption'] ?? '',
            'direction' => 'out',
            'user_id' => $user_id,
            'message_status' => 'Pending',
            'message_type' => 'media',
            'mime_type' => $mime,
            'read_status' => 'Sent',
            'wa_id' => '',
            'media_id' => $mediaId,
            'created_date_time' => now(),
            'updated_date_time' => now(),
            'status' => 'Active'
        ]);

        // Step 4: Send media message to WhatsApp
        $payload = [
            'messaging_product' => 'whatsapp',
            'to' => $contact_number->mobile_no,
            'type' => $mediaType,
            $mediaType => ['id' => $mediaId]
        ];

        if (!empty($validated['caption']) && in_array($mediaType, ['image', 'video', 'document'])) {
            $payload[$mediaType]['caption'] = $validated['caption'];
        }

        $response = Http::withHeaders([
            'Authorization' => "Bearer {$whatsappAccount->access_token}",
            'Content-Type' => 'application/json',
        ])->withOptions(['verify' => false])
            ->post("https://graph.facebook.com/v21.0/{$whatsappAccount->phone_number_id}/messages", $payload);

        if ($response->successful()) {
            $data = $response->json();
            $waId = $data['messages'][0]['id'] ?? null;

            // ✅ Rename file to wa_id
            if ($waId) {
                $newFileName = $waId . '.' . $image->getClientOriginalExtension();
                $newPath = $destinationPath . $newFileName;
                rename($tempPath, $newPath);
            }

            $message->update([
                'wa_id' => $waId,
                'message_status' => 'Sent'
            ]);

            return response()->json([
                'message' => $message,
                'whatsapp_response' => $data
            ], 201);
        } else {
            $message->update(['message_status' => 'Failed']);
            return response()->json([
                'error' => 'Failed to send media message',
                'whatsapp_response' => $response->json()
            ], 400);
        }

    } catch (\Exception $e) {
        return response()->json([
            'error' => 'Exception occurred while sending media message',
            'message' => $e->getMessage()
        ], 500);
    }
}

    public function getContactMessages($contactId)
    {
        $messages = Message::where('contact_id', $contactId)
            ->orderBy('created_date_time', 'asc')
            ->get();

        return response()->json($messages);
    }
    public function getContact()
    {
        $companyId = auth('api')->user()->company_id;

        $contacts = Contacts::where('company_id', $companyId)
            ->get()
            ->map(function ($contact) {

                $lastMessage = \App\Models\Message::where('contact_id', $contact->contact_id)
                    ->orderBy('created_date_time', 'desc')
                    ->first();

                $unreadCount = \App\Models\Message::where('contact_id', $contact->contact_id)
                    ->where('read_status', 'Unread')
                    ->count();

                return [
                    'id' => $contact->contact_id,
                    'name' => $contact->contact_name,
                    'avatar' => $contact->profile ?? 'default_avatar.jpg',
                    'lastMessage' => $lastMessage->message ?? '',
                    'lastMessageOn' => $lastMessage
                        ? \Carbon\Carbon::parse($lastMessage->created_date_time)->format('M d, Y h:i A')
                        : null,
                    'totalUnread' => $unreadCount,
                    'userStatus' => $contact->is_online ? 'online' : 'offline',
                ];
            });

        return response()->json($contacts);
    }
}
