<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Razorpay\Api\Api;
use App\Models\Subscription;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class SubscriptionController extends Controller
{
    protected $razorpay;

    public function __construct()
    {
        $this->razorpay = new Api(
            env('RAZORPAY_KEY'),
            env('RAZORPAY_SECRET')
        );
    }

    public function create(Request $request)
    {
        $request->validate([
            'planId' => 'required|string'
        ]);

        $user = Auth::user();
        $planId = $request->planId;

        // Get plan details based on planId (you might have this in a database)
        $planDetails = $this->getPlanDetails($planId);
        
        if (!$planDetails) {
            return response()->json(['error' => 'Invalid plan'], 400);
        }

        try {
            // Create Razorpay subscription
            $subscription = $this->razorpay->subscription->create([
                'plan_id' => $planDetails['razorpay_plan_id'],
                'customer_notify' => 1,
                'total_count' => $planDetails['billing_cycles'], // for monthly, 12 for annual, etc.
                'notes' => [
                    'user_id' => $user->id,
                    'plan_name' => $planDetails['name']
                ]
            ]);

            // Save to database
            $userSubscription = Subscription::create([
                'user_id' => $user->id,
                'plan_id' => $planId,
                'razorpay_subscription_id' => $subscription->id,
                'status' => 'created',
                'amount' => $planDetails['price'],
                'currency' => 'INR',
                'start_at' => now(),
                'end_at' => now()->addMonth(), // Adjust based on your plan duration
            ]);

            return response()->json([
                'subscription_id' => $subscription->id,
                'razorpay_key' => env('RAZORPAY_KEY'),
                'amount' => $planDetails['price'],
                'name' => $user->name,
                'email' => $user->email,
                'contact' => $user->phone,
            ]);

        } catch (\Exception $e) {
            Log::error('Subscription creation failed: ' . $e->getMessage());
            return response()->json(['error' => 'Subscription creation failed'], 500);
        }
    }

    protected function getPlanDetails($planId)
    {
        // This should come from your database in a real application
        $plans = [
            'basic' => [
                'name' => 'Basic',
                'price' => 49900, // in paise
                'razorpay_plan_id' => env('RAZORPAY_BASIC_PLAN_ID'),
                'billing_cycles' => 1, // monthly
                'duration' => 'month'
            ],
            'pro' => [
                'name' => 'Professional',
                'price' => 99900,
                'razorpay_plan_id' => env('RAZORPAY_PRO_PLAN_ID'),
                'billing_cycles' => 1,
                'duration' => 'month'
            ],
            'enterprise' => [
                'name' => 'Enterprise',
                'price' => 199900,
                'razorpay_plan_id' => env('RAZORPAY_ENTERPRISE_PLAN_ID'),
                'billing_cycles' => 1,
                'duration' => 'month'
            ]
        ];

        return $plans[$planId] ?? null;
    }

    // Add webhook handler for Razorpay subscription events
    public function handleWebhook(Request $request)
    {
        // Verify the webhook signature
        $webhookSecret = env('RAZORPAY_WEBHOOK_SECRET');
        $webhookSignature = $request->header('X-Razorpay-Signature');
        $webhookBody = $request->getContent();
        
        try {
            $this->razorpay->utility->verifyWebhookSignature($webhookBody, $webhookSignature, $webhookSecret);
        } catch (\Exception $e) {
            Log::error('Webhook signature verification failed: ' . $e->getMessage());
            return response()->json(['error' => 'Invalid signature'], 403);
        }

        $payload = $request->all();
        $event = $payload['event'];
        $subscriptionId = $payload['payload']['subscription']['entity']['id'];

        // Find the subscription in your database
        $subscription = Subscription::where('razorpay_subscription_id', $subscriptionId)->first();

        if (!$subscription) {
            return response()->json(['error' => 'Subscription not found'], 404);
        }

        // Handle different events
        switch ($event) {
            case 'subscription.activated':
            case 'subscription.charged':
                $subscription->update(['status' => 'active']);
                break;
                
            case 'subscription.pending':
                $subscription->update(['status' => 'pending']);
                break;
                
            case 'subscription.halted':
                $subscription->update(['status' => 'halted']);
                break;
                
            case 'subscription.cancelled':
                $subscription->update(['status' => 'cancelled']);
                break;
                
            case 'subscription.completed':
                $subscription->update(['status' => 'completed']);
                break;
        }

        return response()->json(['success' => true]);
    }
}