<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;

class WebhookController extends Controller
{
    public function handle(Request $request)
    {
        // ✅ Verification for GET
        if ($request->isMethod('get')) {
            $challenge = $request->get('hub_challenge');
            Log::info('Webhook verification challenge: ' . $challenge);
            return response($challenge, 200);
        }

        // ✅ POST - Webhook Message Received
        if ($request->isMethod('post')) {
            $data = $request->all();
            Log::info('Webhook payload:', $data);

            if (empty($data)) {
                return response()->json(['error' => 'No data received'], 400);
            }

            try {
                $messageBody = $data['entry'][0]['changes'] ?? [];
                if (empty($messageBody)) {
                    return response()->json(['error' => 'Invalid structure'], 422);
                }

                $value = $messageBody[0]['value'];
                $statusData = $value['statuses'][0] ?? $value['messages'][0] ?? [];
                $contactNumber = $statusData['recipient_id']
                    ?? $value['contacts'][0]['wa_id']
                    ?? $statusData['from']
                    ?? null;

                $contactName = $value['contacts'][0]['profile']['name'] ?? 'User';
                $messages = $value['messages'][0]['text']['body'] ?? '';
                $messageType = $value['messages'][0]['type'] ?? 'message';

                // ✅ Check contact or insert new
                $contact = DB::table('contacts')->where('mobile_no', $contactNumber)->first();
                if (!$contact) {
                    $contactId = DB::table('contacts')->insertGetId([
                        'contact_name' => $contactName,
                        'profile' => '20250911061307.png',
                        'mobile_no' => $contactNumber,
                        'status' => 'Active',
                    ]);
                } else {
                    $contactId = $contact->contact_id;
                }

                $messageStatus = $statusData['status'] ?? 'Received';
                $waId = $statusData['id'] ?? $value['messages'][0]['id'] ?? null;
                $direction = 'in';
                $userId = 1;
                $companyId = 1;

                // ✅ Handle media if available
                $imageData = $value['messages'][0][$messageType] ?? [];
                $mediaId = $imageData['id'] ?? null;
                $mimeType = $imageData['mime_type'] ?? null;
                $caption = $imageData['caption'] ?? null;
                $sha256 = $imageData['sha256'] ?? null;

                $message_type = $mimeType ?? 'Chat';
                $fileSize = 0;
                $imageUrl = null;

                // ✅ Download media if present
                if ($mediaId) {
                    $accessToken = env('META_ACCESS_TOKEN', 'YOUR_ACCESS_TOKEN');
                    $graphUrl = "https://graph.facebook.com/v19.0/{$mediaId}?access_token={$accessToken}";

                    $metaResponse = Http::get($graphUrl);
                    if ($metaResponse->successful()) {
                        $metaData = $metaResponse->json();
                        $imageUrl = $metaData['url'] ?? null;
                        $fileSize = $metaData['file_size'] ?? 0;

                        if ($imageUrl) {
                            $mediaResponse = Http::withHeaders([
                                'Authorization' => "Bearer {$accessToken}",
                            ])->get($imageUrl);

                            if ($mediaResponse->successful()) {
                                $extension = match ($mimeType) {
                                    'image/jpeg' => 'jpg',
                                    'application/pdf' => 'pdf',
                                    'video/mp4' => 'mp4',
                                    default => 'ogg',
                                };

                                $fileName = "{$waId}.{$extension}";
                                $path = "file_storage/{$fileName}";
                                Storage::disk('public')->put($path, $mediaResponse->body());
                                Log::info("File stored: {$path}");
                            }
                        }
                    }
                }

                // ✅ Insert into message_log
                DB::table('message_log')->insert([
                    'contact_id' => $contactId,
                    'message_type' => $message_type,
                    'media_id' => $mediaId,
                    'media_url' => $imageUrl,
                    'message' => $messages,
                    'caption' => $caption,
                    'mime_type' => $mimeType,
                    'sha256_hash' => $sha256,
                    'file_size' => $fileSize,
                    'wa_id' => $waId,
                    'message_status' => $messageStatus,
                    'response_json' => json_encode($data),
                    'created_at' => now(),
                ]);

                // ✅ Insert or update message
                $exists = DB::table('messages')->where('wa_id', $waId)->exists();

                if ($exists) {
                    DB::table('messages')->where('wa_id', $waId)->update([
                        'message_status' => $messageStatus,
                    ]);
                } else {
                    DB::table('messages')->insert([
                        'company_id' => $companyId,
                        'type' => 'chat',
                        'contact_id' => $contactId,
                        'direction' => $direction,
                        'message' => $messages,
                        'message_type' => ($messageType != 'text') ? 'media' : 'text',
                        'media_id' => $mediaId,
                        'user_id' => $userId,
                        'message_status' => $messageStatus,
                        'wa_id' => $waId,
                        'status' => 'Active',
                        'created_at' => now(),
                    ]);
                }

                return response()->json(['success' => true], 200);
            } catch (\Exception $e) {
                Log::error('Webhook error: ' . $e->getMessage());
                return response()->json(['error' => $e->getMessage()], 500);
            }
        }

        return response()->json(['message' => 'Invalid request'], 405);
    }
}
